<?php

namespace App\Http\Controllers\Passenger\GroupPassengerMedia;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Helpers\ResponseBuilder;
use App\Http\History\HistoryData;
use App\Http\History\HistoryTables;
use App\Modules\GroupPassengerMedia\Application\Create\CreateGroupPassengerMediaQuery;
use App\Modules\GroupPassengerMedia\Application\Create\CreateGroupPassengerMediaResponse;
use App\Modules\GroupPassengerMedia\Domain\GroupPassengerMediaHasThumbnail;
use App\Modules\GroupPassengerMedia\Domain\GroupPassengerMediaStatus;
use App\Modules\GroupPassengerMedia\Domain\GroupPassengerMediaType;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use App\Modules\GroupPassengerMedia\Infrastructure\GroupPassengerMediaValidators;
use App\Modules\Media\Application\Create\CreateMediaQuery;
use App\Modules\Media\Application\Create\CreateMediaResponse;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Domain\Transaction;

class GroupPassengerMediaInsertController extends Controller
{
    private $domain;
    private HistoryData $history;

    public function __construct(
        private QueryBus $queryBus,
        private Transaction $transaction
    ) {
        $this->domain = "GroupPassengerMedia";
        $this->history = new HistoryData(HistoryTables::GROUP_PASSENGER_MEDIA);
    }

    public function __invoke(Request $request)
    {
        (new LaravelValidator())->validate(
            $request->all(),
            $this->domain,
            GroupPassengerMediaValidators::getInsert()
        );

        /** @var CreateMediaResponse $mediaResponse */
        $mediaResponse = $this->queryBus->ask(new CreateMediaQuery(
            'File',
            'group_provider' . DIRECTORY_SEPARATOR . 'media',
            true
        ));

        $mediaType = GroupPassengerMediaType::fromMediaType($mediaResponse->mediaType);

        $this->transaction->begin();
        /** @var CreateGroupPassengerMediaResponse $response */
        $response = $this->queryBus->ask(new CreateGroupPassengerMediaQuery(
            $mediaResponse->url,
            $mediaResponse->thumbnail,
            $mediaResponse->largeThumbnail,
            '',
            $mediaResponse->extension,
            $mediaResponse->filesize,
            $mediaType->value(),
            $mediaResponse->thumbnail ? GroupPassengerMediaHasThumbnail::VALUE_TRUE : GroupPassengerMediaHasThumbnail::VALUE_FALSE,
            '',
            GroupPassengerMediaStatus::ACTIVE,
            $request->input('Id_GroupPassenger'),
        ));

        $this->transaction->commit();
        $this->history->insertBasic($request, $response->id());
        return ResponseBuilder::Response([
            "Response_Status"           => 200,
            "Response_Code"             => 200,
            "Response_Data"             => ['id' => $response->id()],
            "Response_Domain"           => $this->domain,
            "Response_Message"          => 'Media uploaded successfully',
            "Response_Error_Message"    => '',
            "Response_Error_Reason"     => '',
        ]);
    }
}
