<?php

declare(strict_types=1);

namespace App\Modules\Shared\Domain\ValueObject;

use DateInterval;
use DateTime;
use InvalidArgumentException;
use Stringable;

class DateValueObject implements Stringable
{
    protected static $format = 'Y-m-d';
    protected DateTime $value;

    public function __construct(string $value)
    {
        $this->ensureIsValidDate($value);
        $this->value = DateTime::createFromFormat(self::$format, $value);
    }

    public static function now()
    {
        return new static(date(self::$format));
    }

    public static function reducedNowToPeru()
    {
        return DateTimeValueObject::reduceDatetimeToPeru(DateTimeValueObject::now())->value(DateValueObject::format());
    }

    public function value(): string
    {
        return $this->value->format(self::$format);
    }

    public function year(): int
    {
        return (int)$this->value->format('Y');
    }

    public function month(): int
    {
        return (int)$this->value->format('m');
    }

    public function day(): int
    {
        return (int)$this->value->format('d');
    }

    public function addInterval(string $interval)
    {
        $this->value->add(new DateInterval($interval));
    }

    public function __toString(): string
    {
        return $this->value();
    }

    private function ensureIsValidDate(string $date): void
    {
        $dt = DateTime::createFromFormat(self::$format, $date);
        if (!$dt || $dt->format(self::$format) !== $date) {
            throw new InvalidArgumentException(sprintf('%s does not allow the value %s.', static::class, $date));
        }
    }

    public static function format()
    {
        return self::$format;
    }
}
