<?php

namespace App\Providers\BookingReport\Confirm\Excel;

use App\Http\Helpers\CashMovementHelper;
use App\Providers\ExcelReport\Utils;

class ReportCells
{

    private const BACKGROUNDS = [
        'FFA95B5B',
        'FFCE6F6F',
        'FFE9A135'
    ];

    private const BASIC_BACKGROUND = 'FF707070';
    private const BASIC_COLOR_ACTIVE = 'FF000000';
    private const BASIC_COLOR_INACTIVE = 'FFDF6749';

    private function tableTitle($data)
    {
        $fnPrint = Utils::print($data[0]);
        $fnWidth = Utils::width($data[1]);
        $fnBackground = Utils::background($data[2]);
        $fnColor = Utils::color('FFFFFFFF');
        return $fnColor($fnBackground(Utils::wrap(Utils::top(Utils::border(Utils::bold(Utils::center($fnPrint($fnWidth(Utils::cell())))))))));
    }

    private function tableCellLeft($text, $color, bool $borderBottom)
    {
        $fnPrint = Utils::print($text);
        $fnColor = Utils::color($color);
        if ($borderBottom) {
            $fns = Utils::compose($fnColor, ['App\Providers\ExcelReport\Utils', 'borderBottom']);
        } else {
            $fns = Utils::compose($fnColor);
        }
        return Utils::wrap(Utils::top($fns(Utils::left($fnPrint(Utils::cell())))));
    }

    private function tableRange($rangeValues, $cell, $promptTitle = '', $prompt = '')
    {
        return Utils::rangeValues($rangeValues, $promptTitle, $prompt)($cell);
    }

    private function tableCellCenter($text)
    {
        $fnPrint = Utils::print($text);
        return Utils::wrap(Utils::top(Utils::border(Utils::center($fnPrint(Utils::cell())))));
    }

    private function tableCellRight($text, $color, bool $borderBottom)
    {
        $fnPrint = Utils::print($text);
        $fnColor = Utils::color($color);
        if ($borderBottom) {
            $fns = Utils::compose($fnColor, ['App\Providers\ExcelReport\Utils', 'borderBottom']);
        } else {
            $fns = Utils::compose($fnColor);
        }
        return Utils::wrap(Utils::top($fns(Utils::right($fnPrint(Utils::cell())))));
    }

    private function getTypeDocumentList(ReportBookingDataCollection $data)
    {
        return array_map(fn ($tp) => $tp->TypePayment_Name, $data->typePayment);
    }

    private function getAllRows(ReportBookingDataCollection $data, $length)
    {
        return array_map(fn ($b, $k) => $this->getRow($b, $k, $length, $this->getTypeDocumentList($data)), array_values($data->data()), array_keys($data->data()));
    }

    private function getStatusFrom($status1, $status2, $order)
    {
        foreach ($order as $element) {
            if ($status1 == $element || $status2 == $element) {
                return $element;
            }
        }
    }

    private function groupPackages(array $BookingTourCodes): array
    {
        return array_reduce(
            $BookingTourCodes,
            function ($carry, $bt) {
                if (!isset($carry[$bt->BookingTour_Group])) {
                    $carry[$bt->BookingTour_Group] = (object)[
                        'Element' => $bt,
                        'Tours' => [$bt]
                    ];
                } else {
                    $carry[$bt->BookingTour_Group]->ElementBookingTourPassenger_Status = $this->getStatusFrom(
                        $carry[$bt->BookingTour_Group]->Element->BookingTourPassenger_Status,
                        $bt->BookingTourPassenger_Status,
                        [2, 3, 0, 1]
                    );
                    $carry[$bt->BookingTour_Group]->Tours[] = $bt;
                }
                return $carry;
            },
            []
        );
    }

    private function concatPackages(array $packages): string
    {
        $specifyData = ['NO VIAJO', 'ANULADO', 'ACTIVO', 'DIO ALCANCE'];

        return implode("\n", array_map(function ($t) use ($specifyData) {
            $specify = $specifyData[$t->Element->BookingTourPassenger_Status] ?? '';
            return $t->Element->BookingTour_Name
                . ' - '
                . $specify
                . ' (' . date('d-m-Y', strtotime($t->Element->BookingTour_DateStart)) . ')';
        }, $packages));
    }

    private function concatTours(array $packages): string
    {
        $specifyData = ['NO VIAJO', 'ANULADO', 'ACTIVO', 'DIO ALCANCE'];

        return implode("\n", array_map(function ($pack) use ($specifyData) {
            return $pack->Element->Package_Code
                . "\n"
                . implode(
                    "\n",
                    array_map(
                        fn($t) => $t->BookingTour_TourName
                            . ' - '
                            . ($specifyData[$t->BookingTourPassenger_Status] ?? '')
                            . ' (' . date('d-m-Y', strtotime($t->BookingTour_DateStart)) . ')',
                        $pack->Tours
                    )
                );
        }, $packages));
    }

    private function getRow(ReportBookingData $booking, $key, $length, $typePayments)
    {
        $color = $booking->BookingTourPassenger_Status > 0 ? self::BASIC_COLOR_ACTIVE : self::BASIC_COLOR_INACTIVE;

        $packages = $this->groupPackages($booking->BookingTourCodes);
        return array_merge(
            [
                $this->tableCellLeft($key + 1, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Booking_Code, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Booking_Date, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Contact_Name, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->User_Email, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Booking_Type, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Admin_Name . ' ' . $booking->Admin_LastName, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Package_Duration, $color, $booking->LastPassenger),
                $this->tableCellLeft($this->concatPackages($packages), $color, $booking->LastPassenger),
                $this->tableCellLeft($this->concatTours($packages), $color, $booking->LastPassenger),
                $this->tableCellRight($booking->Booking_DateStart, $color, $booking->LastPassenger),
                $this->tableCellRight($booking->Passenger_CheckedInStatus == 2 ? 'SÍ' : '', $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Passenger_FullName, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->BookingTourPassenger_Status > 0 ? 'ACTIVO' : 'INACTIVO', $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->Passenger_NoDocument, $color, $booking->LastPassenger),
                $this->tableCellLeft($booking->UserCountry_Name, $color, $booking->LastPassenger),
                $this->tableCellRight($booking->Passenger_Gender == 1 ? 'M' : 'F', $color, $booking->LastPassenger),
            ],
            $this->generateCashCells($booking->payments, $length, $color, $typePayments, $booking->LastPassenger),
            [
                $this->tableCellRight($booking->Payment_Total, $color, $booking->LastPassenger),
                $this->tableCellRight($booking->FirstPassenger ? $booking->CashMovement_Total : '', $color, $booking->LastPassenger),
            ],
        );
    }

    public function generateCashCells(PaymentDataCollection $cashMovementCollection, $allLength, $color, $typePayments, $lastPassenger)
    {
        $titles = [];
        $length = count($cashMovementCollection);
        for ($i = 0; $i < $allLength; $i++) {
            if ($i < $length) {
                $titles = array_merge($titles, $this->generateCashCell($cashMovementCollection[$i], $color, $typePayments, $lastPassenger));
            } else {
                $titles = array_merge($titles, $this->generateEmptyCashCell($color, $typePayments, $lastPassenger));
            }
        }
        return $titles;
    }

    public function generateCashCell(PaymentData $cashMovement, $color, $typePayments, $lastPassenger)
    {
        return [
            $this->tableRange(
                $typePayments,
                $this->tableCellLeft($cashMovement->TypePayment_Name, $color, $lastPassenger),
                'Elegir de la lista',
                'Elegir de la lista'
            ),
            $this->tableCellLeft(CashMovementHelper::changeReceiptNumber($cashMovement), $color, $lastPassenger),
            $this->tableCellLeft($cashMovement->CashMovement_DatePayment, $color, $lastPassenger),
            $this->tableCellRight($cashMovement->CashMovement_Amount, $color, $lastPassenger),
        ];
    }

    public function generateEmptyCashCell($color, $typePayments, $lastPassenger)
    {
        return [
            $this->tableRange(
                $typePayments,
                $this->tableCellLeft('', $color, $lastPassenger),
                'Elegir de la lista',
                'Elegir de la lista'
            ),
            $this->tableCellLeft('', $color, $lastPassenger),
            $this->tableCellLeft('', $color, $lastPassenger),
            $this->tableCellRight('', $color, $lastPassenger),
        ];
    }

    public function generateCashTitles($length)
    {
        $titles = [];
        for ($i = 0; $i < $length; $i++) {
            $titles[] = $this->generateCashTitle($i);
        }

        $titles[] = [
            $this->tableTitle(['MONTO TOTAL', 10, self::BASIC_BACKGROUND]),
            $this->tableTitle(['MONTO TOTAL', 10, self::BASIC_BACKGROUND])
        ];

        return $titles;
    }

    public function generateCashTitle($i)
    {
        return array_map([$this, 'tableTitle'], [
            ['TIPO', 14, self::BACKGROUNDS[$i % count(self::BACKGROUNDS)]],
            ['NUM. PAGO', 20, self::BACKGROUNDS[$i % count(self::BACKGROUNDS)]],
            ['FECHA', 12, self::BACKGROUNDS[$i % count(self::BACKGROUNDS)]],
            ['P' . ($i + 1), 8, self::BACKGROUNDS[$i % count(self::BACKGROUNDS)]]
        ]);
    }

    public function run(ReportBookingDataCollection $data, $metadata)
    {
        $length = 0;
        foreach ($data->data() as $d) {
            $length = $length > count($d->payments) ? $length : count($d->payments);
        }
        return array_merge(
            [
                array_merge(
                    array_map([$this, 'tableTitle'], [
                        ['N°', 5, self::BASIC_BACKGROUND],
                        ['BOOKING', 12, self::BASIC_BACKGROUND],
                        ['REGISTRO', 12, self::BASIC_BACKGROUND],
                        ['CONTACTO', 16, self::BASIC_BACKGROUND],
                        ['EMAIL', 16, self::BASIC_BACKGROUND],
                        ['TIPO', 12, self::BASIC_BACKGROUND],
                        ['VENDEDOR', 12, self::BASIC_BACKGROUND],
                        ['DÍAS', 6, self::BASIC_BACKGROUND],
                        ['PAQUETES', 35, self::BASIC_BACKGROUND],
                        ['TOURS', 35, self::BASIC_BACKGROUND],
                        ['F. SALIDA', 12, self::BASIC_BACKGROUND],
                        ['CONF.', 7, self::BASIC_BACKGROUND],
                        ['PASAJERO', 20, self::BASIC_BACKGROUND],
                        ['ESTADO', 12, self::BASIC_BACKGROUND],
                        ['DOCUMENTO', 14, self::BASIC_BACKGROUND],
                        ['PAIS', 24, self::BASIC_BACKGROUND],
                        ['S', 5, self::BASIC_BACKGROUND],
                    ]),
                    ...$this->generateCashTitles($length),
                ),
            ],
            $this->getAllRows($data, $length)
        );
    }
}
