<?php

namespace App\Http\Controllers\Owner\Package;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Helpers\ResponseBuilder;
use App\Http\History\HistoryData;
use App\Http\History\HistoryTables;
use App\Modules\HtlCamp\Application\HtlCampIndexer;
use App\Modules\HtlCamp\Domain\IdHtlCamp;
use App\Modules\HtlCamp\Infraestructure\MysqlHtlCampRepository;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use App\Modules\HtlCampBooking\Application\Update\UpdateHtlCampBookingQuery;
use App\Modules\HtlCampBooking\Infrastructure\HtlCampBookingValidators;
use App\Modules\HtlCampSchedule\Application\Availability\AvailabilityHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Application\HtlCampScheduleResponses;
use App\Modules\HtlCampSchedule\Application\List\ListHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Application\Order\CalendarGenerator;
use App\Modules\HtlCampSchedule\Application\Order\MapSchedule;
use App\Modules\HtlCampSchedule\Application\Order\OrderHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Application\Order\OrderHtlCampScheduleResponse;
use App\Modules\HtlCampSchedule\Application\ReloadAmount\ReloadAmountHtlCampScheduleQuery;
use App\Modules\Shared\Application\SelectHandler;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Domain\DateEnd;
use App\Modules\Shared\Domain\DateStart;
use App\Modules\Shared\Domain\Transaction;
use App\Modules\Shared\Exceptions\BasicRequestException;
use App\Modules\TourHtlCamp\Application\TourHtlCampFinder;
use App\Modules\TourHtlCamp\Domain\THCIdTour;
use App\Modules\TourHtlCamp\Infraestructure\MysqlTourHtlCampRepository;
use Illuminate\Support\Facades\DB;

class PackageAvailabilityController extends Controller
{
    private $domain;

    public function __construct(
        private QueryBus $queryBus,
        private CalendarGenerator $generator
    ) {
        $this->domain = "HtlCampBooking";
    }

    public function __invoke(Request $request)
    {
        $validators = [
            'Id_Package' => 'required|int',
            'Date_Start' => 'required|string|date',
            'Date_End' => 'required|string|date'
        ];
        (new LaravelValidator())->validate($request->all(), $this->domain, $validators);
        $dateStart = new DateStart($request->input('Date_Start'));
        $dateEnd = new DateEnd($request->input('Date_End'));

        $idPackage = $request->input('Id_Package');
        $package = $this->getPackageIndex($idPackage);
        $tourList = $this->getTourList($idPackage);

        foreach ($tourList as $i => $tour) {
            $tour->campList = $this->getTourHtlCampList($tour->Id_Tour);
            $tour->day = $i === 0 ? 1 : $tourList[$i - 1]->day + $tourList[$i - 1]->Tour_Duration;
            foreach ($tour->campList as $camp) {
                $camp->day = $tour->day + $camp->TourHtlCamp_Day - 1;
            }
        }
        $generated = $this->generator->__invoke($dateStart, $dateEnd);
        $days = $this->getInitialAndLastDay($tourList);
        $dateStart->addInterval('P' . ($days['InitialDay'] - 1) . 'D');
        $dateEnd->addInterval('P' . ($days['LastDay'] - 1) . 'D');

        $calendar = $this->getCalendars($generated, $tourList, $dateStart->value(), $dateEnd->value());

        return ResponseBuilder::Response([
            "Response_Status"           => 200,
            "Response_Code"             => 200,
            "Response_Data"             => $calendar,
            "Response_Domain"           => $this->domain,
            "Response_Message"          => 'Lista',
            "Response_Error_Message"    => '',
            "Response_Error_Reason"     => '',
        ]);
    }

    private function getCalendars($calendar, $tourList, $dateStart, $dateEnd)
    {
        foreach ($calendar as $day) {
            foreach ($tourList as $tour) {
                foreach ($tour->campList as $camp) {
                    $campValue = $this->calendarCamp($camp, $dateStart, $dateEnd);
                    $availability = $campValue->getAvailability($day->date);
                    if ($day->availability === null) {
                        $day->availability = $availability;
                    } else {
                        $day->availability = $availability < $day->availability ? $availability : $day->availability;
                    }
                }
            }
        }
        return $calendar;
    }

    private function calendarCamp($camp, $dateStart, $dateEnd)
    {
        $indexResponse = (new HtlCampIndexer(new MysqlHtlCampRepository))->__invoke(new IdHtlCamp($camp->Id_HtlCamp));
        /** @var HtlCampScheduleResponses $response */
        $response = $this->queryBus->ask(new ListHtlCampScheduleQuery($camp->Id_HtlCamp, $dateStart . ' 00:00:00', $dateEnd . ' 23:59:59'));
        /** @var OrderHtlCampScheduleResponse $order */
        $order = $this->queryBus->ask(new OrderHtlCampScheduleQuery(
            $dateStart . ' 00:00:00',
            $dateEnd . ' 23:59:59',
            $response->data(),
            $indexResponse['Response_Data'][0]
        ));
        return new MapSchedule($order->response);
    }

    private function getInitialAndLastDay($tourList)
    {
        $initialDay = 1;
        $lastDay = 1;
        foreach ($tourList as $i => $tour) {
            if ($i !== 0) {
                $lastDay = $tour->day > $lastDay ? $tour->day : $lastDay;
            }

            foreach ($tour->campList as $camp) {
                $day = $camp->TourHtlCamp_Day + $tour->day - 1;
                $lastDay = $day > $lastDay ? $day : $lastDay;
                $initialDay = $day < $initialDay ? $day : $initialDay;
            }
        }

        return ['InitialDay' => $initialDay, 'LastDay' => $lastDay];
    }

    private function getPackageIndex($idPackage)
    {
        $package = $this->query('call sp_package_index(?)', [$idPackage]);
        if (!count($package)) {
            throw new BasicRequestException(404, 404, 'No existe el paquete');
        }
        return $package;
    }

    private function getTourList($idPackage)
    {
        return $this->query('call sp_package_tour_list(?)', [$idPackage]);
    }

    private function query($query, $bindings)
    {
        return (new SelectHandler())->handle(DB::select($query, $bindings));
    }

    private function getTourHtlCampList($idTour)
    {
        $response = (new TourHtlCampFinder(new MysqlTourHtlCampRepository))->__invoke(new THCIdTour($idTour), 'active');
        return $response['Response_Data'];
    }
}
