<?php

namespace App\Http\Controllers\Owner\HtlCampSchedule;

use App\Http\Controllers\Controller;
use App\Http\Helpers\ResponseBuilder;
use App\Modules\HtlCamp\Application\HtlCampFinder;
use App\Modules\HtlCamp\Application\HtlCampIndexer;
use App\Modules\HtlCamp\Domain\IdHtlCamp;
use App\Modules\HtlCamp\Infraestructure\MysqlHtlCampRepository;
use App\Modules\HtlCampSchedule\Application\HtlCampScheduleResponses;
use App\Modules\HtlCampSchedule\Application\List\ListHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Application\Order\OrderHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Application\Order\OrderHtlCampScheduleResponse;
use App\Modules\HtlCampSchedule\Application\OrderList\OrderListHtlCampScheduleQuery;
use App\Modules\HtlCampSchedule\Infrastructure\HtlCampScheduleValidators;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Domain\Transaction;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use Illuminate\Http\Request;

class HtlCampScheduleListController extends Controller
{
    private $domain;

    public function __construct(
        private QueryBus $queryBus,
        private Transaction $transaction
    ) {
        $this->domain = "HtlCampSchedule";
    }

    public function __invoke(Request $request, $Id_HtlCamp)
    {
        (new LaravelValidator())->validate($request->all(), $this->domain, HtlCampScheduleValidators::LIST);

        if ($Id_HtlCamp == 0) {
            return $this->getAll($request);
        } else {
            return $this->getIndex($request, $Id_HtlCamp);
        }
    }

    private function getAll(Request $request)
    {
        $response = (new HtlCampFinder(new MysqlHtlCampRepository))->__invoke('active', '');

        $this->transaction->begin();
        $list = $response['Response_Data'];
        $schedules = [];
        foreach ($list as $value) {
            /** @var HtlCampScheduleResponses $response */
            $response = $this->queryBus->ask(new ListHtlCampScheduleQuery(
                $value->Id_HtlCamp,
                $request->input('DateStart') . ' 00:00:00',
                $request->input('DateEnd') . ' 23:59:59',
            ));

            /** @var OrderHtlCampScheduleResponse $orderer */
            $response = $this->queryBus->ask(new OrderHtlCampScheduleQuery(
                $request->input("DateStart") . ' 00:00:00',
                $request->input("DateEnd") . ' 23:59:59',
                $response->data(),
                $value
            ))->response;

            foreach ($response as $i => $date) {
                if (!isset($schedules[$i])) {
                    $schedules[$i] = (object)[
                        'date' => $date->date,
                        'list' => [$date]
                    ];
                } else {
                    $schedules[$i]->list[] = $date;
                }
            }
        }

        $this->transaction->commit();

        return ResponseBuilder::Response([
            "Response_Status"           => 200,
            "Response_Code"             => 200,
            "Response_Data"             => $schedules,
            "Response_Domain"           => $this->domain,
            "Response_Message"          => 'Listado con éxito',
            "Response_Error_Message"    => '',
            "Response_Error_Reason"     => '',
        ]);
    }

    private function getIndex(Request $request, $Id_HtlCamp)
    {
        $this->transaction->begin();
        $indexResponse = (new HtlCampIndexer(new MysqlHtlCampRepository))->__invoke(new IdHtlCamp($Id_HtlCamp));

        /** @var HtlCampScheduleResponses $response */
        $response = $this->queryBus->ask(new ListHtlCampScheduleQuery(
            $Id_HtlCamp,
            $request->input('DateStart') . ' 00:00:00',
            $request->input('DateEnd') . ' 23:59:59',
        ));

        /** @var OrderHtlCampScheduleResponse $orderer */
        $orderer = $this->queryBus->ask(new OrderListHtlCampScheduleQuery(
            $request->input("DateStart") . ' 00:00:00',
            $request->input("DateEnd") . ' 23:59:59',
            $response->data(),
            $indexResponse['Response_Data'][0]
        ));

        $this->transaction->commit();

        return ResponseBuilder::Response([
            "Response_Status"           => 200,
            "Response_Code"             => 200,
            "Response_Data"             => $orderer->response,
            "Response_Domain"           => $this->domain,
            "Response_Message"          => 'Listado con éxito',
            "Response_Error_Message"    => '',
            "Response_Error_Reason"     => '',
        ]);
    }
}
