<?php

namespace App\Http\Controllers\Owner;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Helpers\ResponseBuilder;
use App\Http\Helpers\ParamsValidator;
use App\Http\History\HistoryData;
use App\Http\History\HistoryTables;
use App\Modules\Admin\Application\Create\AdminInserterQuery;
use App\Modules\Admin\Application\Create\AdminInserterResponse;
use App\Modules\Admin\Application\Index\AdminIndexerQuery;
use App\Modules\Admin\Application\Index\AdminIndexerResponse;
use App\Modules\Admin\Application\Update\AdminUpdaterQuery;
use App\Modules\Admin\Application\Update\AdminUpdaterResponse;
use App\Modules\Admin\Infrastructure\AdminValidators;
use App\Modules\Image\Domain\ImageThumbnailData;
use App\Modules\Image\Infraestructure\BasicConfigImageStorage;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Infraestructure\LaravelExceptionHandler;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use Illuminate\Support\Facades\DB;
use Throwable;

class Admin extends Controller
{
    private $oDomain;
    /** @var HistoryData */
    private $history;

    public function __construct(
        private QueryBus $queryBus
    ) {
        $this->oDomain = "Admin Session";
        $this->history = new HistoryData(HistoryTables::ADMIN);
    }

    public function Insert(Request $request)
    {
        (new LaravelValidator())->validate($request->all(), $this->oDomain, AdminValidators::insert());

        $thumbnail = new ImageThumbnailData(
            config("var.THUMB_W"),
            config("var.THUMB_H")
        );

        $image = (new BasicConfigImageStorage($request))->storeImage('admin' . DIRECTORY_SEPARATOR . 'profile', 'Admin_ImgProfile', $thumbnail);

        /** @var AdminInserterResponse $response */
        $response = $this->queryBus->ask(new AdminInserterQuery(
            Str::title($request->input("Admin_Name")),
            Str::title($request->input("Admin_LastName")),
            Str::upper($request->input("Admin_Abrv")),
            $request->input("Admin_NoDocument"),
            $request->input("Admin_Email"),
            $request->input("Admin_EmailAddress"),
            $request->input("Admin_Address"),
            $request->input("Admin_Phone"),
            $request->input("Admin_Color"),
            $request->input("Admin_IsAssignable"),
            isset($image) ? $image->getUrl() : '',
            isset($image) ? $image->getThumbnail() : '',
            $request->input("Admin_Status"),
            $request->input("Id_TypeDocument"),
            $request->input("Id_AdminRole"),
            $request->input("Id_Country"),
            $request->input("Id_State"),
            $request->input("Id_City")
        ));

        $data = ['Id_Admin' => $response->response()[0]->Response_Id];

        $this->history->insert(
            [$request->all()],
            $request,
            $response->response()[0]->Response_Id
        );
        $this->history->insertBasic($request);

        $oResponse["Response_Status"]           = $response->response()[0]->Response_Status;
        $oResponse["Response_Code"]             = $response->response()[0]->Response_Code;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $response->response()[0]->Response_Message;
        $oResponse["Response_Data"]             = $data;
        $oResponse["Response_Error_Message"]    = $response->response()[0]->Response_Message;
        $oResponse["Response_Error_Reason"]     = $response->response()[0]->Response_Reason;
        return ResponseBuilder::Response($oResponse);
    }

    private function indexOrEmpty(AdminIndexerResponse $response, $key)
    {
        if (count($response->response())) {
            return $response->response()[0]->{$key};
        } else {
            return '';
        }
    }

    private function unlinkOnSet($fileValue, $file)
    {
        if (isset($fileValue) && $fileValue !== '') {
            unlink($file);
        }
    }

    public static function appendPath($admin)
    {
        if ($admin->Admin_ImgProfile) {
            $admin->Admin_ImgProfile = config('var.PATH_PUBLIC') . $admin->Admin_ImgProfile;
        }
        if ($admin->Admin_ImgProfile_Thumb) {
            $admin->Admin_ImgProfile_Thumb = config('var.PATH_PUBLIC') . $admin->Admin_ImgProfile_Thumb;
        }
        return $admin;
    }

    public function Update(Request $request)
    {
        (new LaravelValidator())->validate($request->all(), $this->oDomain, AdminValidators::update());

        $this->history->obtainOld($request->input('Id_Admin'));

        $thumbnail = new ImageThumbnailData(
            config("var.THUMB_W"),
            config("var.THUMB_H")
        );

        $image = (new BasicConfigImageStorage($request))->storeImage('admin' . DIRECTORY_SEPARATOR . 'profile', 'Admin_ImgProfile', $thumbnail);

        /** @var AdminIndexerResponse $indexResponse */
        $indexResponse = $this->queryBus->ask(new AdminIndexerQuery($request->input("Id_Admin")));
        $imgProfile = $this->indexOrEmpty($indexResponse, 'Admin_ImgProfile');
        $imgProfileThumb = $this->indexOrEmpty($indexResponse, 'Admin_ImgProfile_Thumb');

        /** @var AdminUpdaterResponse $response */
        $response = $this->queryBus->ask(new AdminUpdaterQuery(
            $request->input("Id_Admin"),
            Str::title($request->input("Admin_Name")),
            Str::title($request->input("Admin_LastName")),
            Str::upper($request->input("Admin_Abrv")),
            $request->input("Admin_NoDocument"),
            $request->input("Admin_Email"),
            $request->input("Admin_EmailAddress"),
            $request->input("Admin_Address"),
            $request->input("Admin_Phone"),
            $request->input("Admin_Color"),
            $request->input("Admin_IsAssignable"),
            isset($image) ? $image->getUrl() : $imgProfile,
            isset($image) ? $image->getThumbnail() : $imgProfileThumb,
            $request->input("Admin_Status"),
            $request->input("Id_TypeDocument"),
            $request->input("Id_AdminRole"),
            $request->input("Id_Country"),
            $request->input("Id_State"),
            $request->input("Id_City")
        ));

        // if ($image) {
        //     $this->unlinkOnSet($imgProfile, config('var.PATH_PRIVATE') . $imgProfile);
        //     $this->unlinkOnSet($imgProfileThumb, config('var.PATH_PRIVATE') . $imgProfileThumb);
        // }

        $this->history->updateBasic($request->input("Id_Admin"), $request);

        $oResponse["Response_Status"]           = $response->response()[0]->Response_Status;
        $oResponse["Response_Code"]             = $response->response()[0]->Response_Code;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $response->response()[0]->Response_Message;
        $oResponse["Response_Data"]             = null;
        $oResponse["Response_Error_Message"]    = $response->response()[0]->Response_Message;
        $oResponse["Response_Error_Reason"]     = $response->response()[0]->Response_Reason;


        return ResponseBuilder::Response($oResponse);
    }

    public function Delete(Request $Request, $Id)
    {
        $oResponse  = array();

        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($Id, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {

            $this->history->obtainOld($Id);

            $oParam = array($Id);
            $oData  = DB::select('call sp_admin_delete(?)', $oParam);

            //  RESPONSE
            if ($oData[0]->Response_Success == 1) {
                $this->history->deleteBasic($Id, $Request);
            } else {
            }

            $oResponse["Response_Status"]           = $oData[0]->Response_Status;
            $oResponse["Response_Code"]             = $oData[0]->Response_Code;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $oData[0]->Response_Message;
            $oResponse["Response_Data"]             = null;
            $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
            $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function IndexToken(Request $Request)
    {
        $oResponse  = array();

        $oParam = array($Request->header("Token"));
        $oData  = DB::select('call sp_admin_index_token(?)', $oParam);
        $oData = array_map(fn ($admin) => self::appendPath($admin), $oData);

        $oResponse["Response_Status"]           = 200;
        $oResponse["Response_Code"]             = 200;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $this->oDomain . " Index";
        $oResponse["Response_Data"]             = $oData;
        $oResponse["Response_Error_Message"]    = "";
        $oResponse["Response_Error_Reason"]     = "";
        return ResponseBuilder::Response($oResponse);
    }

    public function Index(Request $Request, $Id)
    {
        $oResponse  = array();

        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($Id, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = array($Id);
            $oData  = DB::select('call sp_admin_index(?)', $oParam);
            $oData = array_map(fn ($admin) => self::appendPath($admin), $oData);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " Index";
            $oResponse["Response_Data"]             = $oData;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function List(Request $Request, $Id)
    {
        $oResponse  = array();
        $oResponse = ParamsValidator::Validate_Action($Id, $this->oDomain);

        if ($oResponse["Response_Code"] == 200) {
            $oParam = array($Id);
            $oData  = DB::select('call sp_admin_list(?)', $oParam);
            $oData = array_map(fn ($admin) => self::appendPath($admin), $oData);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " list " . $Id;
            $oResponse["Response_Data"]             = $oData;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function Selector(Request $Request, $Id)
    {
        if (is_numeric($Id)) {
            return SELF::Index($Request, $Id);
        } else {
            return SELF::List($Request, $Id);
        }
    }

    public function Password(Request $Request)
    {
        $oResponse  = array();
        $oValParams = array(
            'Admin_Password'        => 'required|string|max:250',
            'Admin_Password_New'    => 'required|string|max:250'
        );


        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Request($Request, $oValParams, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = array(
                $Request->header("Token"),
                $Request->input("Admin_Password"),
                $Request->input("Admin_Password_New")
            );
            $oData = DB::select('call sp_admin_update_current_password(?,?,?)', $oParam);

            //  RESPONSE
            if ($oData[0]->Response_Success == 1) {
            } else {
            }

            $oResponse["Response_Status"]           = $oData[0]->Response_Status;
            $oResponse["Response_Code"]             = $oData[0]->Response_Code;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $oData[0]->Response_Message;
            $oResponse["Response_Data"]             = null;
            $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
            $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function ChangePassword(Request $request)
    {
        try {
            return $this->ChangePasswordFunction($request);
        } catch (Throwable $ex) {
            return (new LaravelExceptionHandler())->handle($ex);
        }
    }

    public function ChangePasswordFunction(Request $Request)
    {
        $oResponse  = array();

        (new LaravelValidator)->validate($Request->all(), $this->oDomain, AdminValidators::CHANGE_PASSWORD);

        $oParam = [
            $Request->input('Id_Admin'),
            $Request->header('Token'),
            $Request->input("Admin_Password"),
            $Request->input("Admin_Password_New")
        ];

        $oData = DB::select('call sp_admin_update_password(?,?,?,?)', $oParam);

        //  RESPONSE
        if ($oData[0]->Response_Success == 1) {
        } else {
        }

        $oResponse["Response_Status"]           = $oData[0]->Response_Status;
        $oResponse["Response_Code"]             = $oData[0]->Response_Code;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $oData[0]->Response_Message;
        $oResponse["Response_Data"]             = null;
        $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
        $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;


        return ResponseBuilder::Response($oResponse);
    }
}
